<?php
/**
 * Plugin Update Checker Library 4.11
 * http://w-shadow.com/
 *
 * Copyright 2021 Janis Elsts
 *
 * Licensed under the MIT license.
 * See the license.txt file for details.
 */

if ( !class_exists('Puc_v4_Factory', false) ):

	/**
	 * A factory that builds update checker instances.
	 *
	 * When multiple versions of the same class have been loaded (e.g. Plugin A loads 1.2, Plugin B loads 1.3),
	 * this factory will always use the latest available version. Register class versions by calling
	 * Puc_v4_Factory::addVersion().
	 *
	 * At the moment it can only build instances of the Puc_v4p11_Plugin_UpdateChecker class. Other classes are planned.
	 */
	class Puc_v4_Factory {
		protected static $classVersions = array();
		protected static $sorted = false;

		protected static $myMajorVersion = '';
		protected static $latestCompatibleVersion = '';

		/**
		 * A wrapper around the plugin update checker constructor.
		 *
		 * @see Puc_v4p11_Plugin_UpdateChecker::__construct()
		 *
		 * @param $metadataUrl
		 * @param $pluginFile
		 * @param string $slug
		 * @param int $checkPeriod
		 * @param string $optionName
		 * @param string $muPluginFile
		 * @return Puc_v4p11_Plugin_UpdateChecker
		 */
		public static function buildUpdateChecker($metadataUrl, $pluginFile, $slug = '', $checkPeriod = 12, $optionName = '', $muPluginFile = '') {
			$majorVersion = self::getMajorVersion($pluginFile);
			$class = self::getLatestClassVersion($majorVersion);

			if ( $class === null ) {
				//This can happen if the factory is loaded but the library classes are not.
				//This is not ideal, but the library is probably not installed correctly then.
				//See: https://github.com/YahnisElsts/plugin-update-checker/issues/196
				if ( class_exists('Puc_v4p11_Plugin_UpdateChecker', false) ) {
					$class = 'Puc_v4p11_Plugin_UpdateChecker';
				} else {
					trigger_error(
						'Plugin Update Checker: Could not find a PUC class for the major version "' . $majorVersion . '".',
						E_USER_ERROR
					);
					return null;
				}
			}

			return new $class($metadataUrl, $pluginFile, $slug, $checkPeriod, $optionName, $muPluginFile);
		}

		/**
		 * Get the major PUC version that the plugin is using.
		 *
		 * @param string $pluginFile
		 * @return string
		 */
		protected static function getMajorVersion($pluginFile) {
			if ( self::$myMajorVersion === '' ) {
				//Find the major version of *this* factory class.
				$name = __CLASS__;
				if ( preg_match('/_v(\d+)_Factory$/', $name, $matches) ) {
					self::$myMajorVersion = $matches[1];
				}
			}
			return self::$myMajorVersion;
		}

		/**
		 * Get the latest available class version for a given major version.
		 *
		 * @param string $majorVersion
		 * @return string|null
		 */
		public static function getLatestClassVersion($majorVersion) {
			if ( self::$latestCompatibleVersion === '' ) {
				self::sortVersions();

				//Find the latest version that's compatible with the major version of this factory.
				self::$latestCompatibleVersion = null;
				foreach (self::$classVersions as $version => $class) {
					if ( substr($version, 0, strlen($majorVersion)) === $majorVersion ) {
						self::$latestCompatibleVersion = $class;
						break;
					}
				}
			}
			return self::$latestCompatibleVersion;
		}

		/**
		 * Sort available class versions in descending order (i.e. latest version first).
		 */
		protected static function sortVersions() {
			//If it's already sorted, skip.
			if ( self::$sorted ) {
				return;
			}
			krsort(self::$classVersions, SORT_NATURAL);
			self::$sorted = true;
		}

		/**
		 * Register a version of a class.
		 *
		 * @param string $class The class name.
		 * @param string $version The version number, e.g. "4.1.2".
		 */
		public static function addVersion($class, $version) {
			if ( isset(self::$classVersions[$version]) ) {
				return;
			}
			self::$classVersions[$version] = $class;
			self::$sorted = false;
			self::$latestCompatibleVersion = '';
		}
	}

endif;

//Register the factory itself.
Puc_v4_Factory::addVersion('Puc_v4_Factory', '4.11');

//Register the update checker classes.
Puc_v4_Factory::addVersion('Puc_v4p11_Plugin_UpdateChecker', '4.11');
Puc_v4_Factory::addVersion('Puc_v4p11_Vcs_PluginUpdateChecker', '4.11');

//Load the rest of the library.
if ( file_exists(dirname(__FILE__) . '/load-v4p11.php') ) {
	include_once dirname(__FILE__) . '/load-v4p11.php';
}
