<?php
/**
 * Plugin Name:       WP Social Connect
 * Plugin URI:        https://ehitb.com/wp-social-connect
 * Description:       Add beautiful, customizable, and floating social media icons to your WordPress site.
 * Version:           1.2.0
 * Author:            Ekramul Hassan Infotech Bangladesh
 * Author URI:        https://ehitb.com
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       wp-social-connect
 * Domain Path:       /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

// Plugin version.
define( 'WPSC_VERSION', '1.2.0' );

// Include the update checker library.
require_once plugin_dir_path( __FILE__ ) . 'updater/plugin-update-checker.php';
$wpsc_update_checker = Puc_v4_Factory::buildUpdateChecker(
    'https://ehitb.com/wp-social-connect-update.json',
    __FILE__,
    'wp-social-connect'
);

/**
 * Activation hook to set default options.
 */
function wpsc_activate() {
    if ( false === get_option( 'wpsc_settings' ) ) {
        add_option( 'wpsc_settings', array(
            'facebook_url' => '', 'twitter_url' => '', 'instagram_url' => '', 'linkedin_url' => '', 'youtube_url' => '', 'whatsapp_url' => '', 'telegram_url' => '', 'email_url' => '', 'call_url' => '',
            'custom1_url' => '', 'custom1_label' => '', 'custom1_icon' => '',
            'custom2_url' => '', 'custom2_label' => '', 'custom2_icon' => '',
            'custom3_url' => '', 'custom3_label' => '', 'custom3_icon' => '',
            'position' => 'bottom-left',
            'icon_size' => 44,
            'icon_shape' => 'circle',
            'color_scheme' => 'brand',
            'custom_bg_color' => '#333333',
            'custom_icon_color' => '#ffffff',
            'display_mode' => 'all',
            'toggle_icon' => 'fas fa-phone',
            'toggle_direction' => 'upwards',
            'force_display' => '0',
        ) );
    }
}
register_activation_hook( __FILE__, 'wpsc_activate' );

/**
 * Add the admin menu page.
 */
function wpsc_add_admin_menu() {
    add_menu_page( 'WP Social Connect Settings', 'WP Social Connect', 'manage_options', 'wp-social-connect', 'wpsc_admin_page_html', 'dashicons-share', 80);
}
add_action( 'admin_menu', 'wpsc_add_admin_menu' );

/**
 * Enqueue admin scripts and styles.
 */
function wpsc_admin_enqueue_scripts($hook_suffix) {
    if ('toplevel_page_wp-social-connect' !== $hook_suffix) {
        return;
    }
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');

    $css = "
        .wpsc-wrap .wpsc-header { background: #ebecf5; border-bottom: 1px solid #ddd;border-radius:30px 30px 0px 0px; padding: 20px; margin-left: -20px; margin-bottom: 20px; display: flex; align-items: center; }
        .wpsc-wrap .wpsc-header img { width: 200px !important; max-width: 200px !important; height: auto; margin-right: 20px; }
        .wpsc-wrap .wpsc-powered-by { font-size: 12px; color: #fff; margin-top: 5px; }
        .wpsc-wrap .wpsc-powered-by a { text-decoration: none; }
        .wpsc-wrap .form-table th { width: 200px; }
        .wpsc-wrap .form-table .description { font-style: italic; }
        .wpsc-wrap h3 { border-bottom: 1px solid #ddd; padding-bottom: 10px; margin-top: 30px; }
        .wpsc-wrap .wpsc-custom-field-group { display: block; border-left: 3px solid #eee; padding-left: 15px; margin-top: 10px; }
        .wpsc-wrap .wpsc-custom-field-group p { margin-bottom: 15px; }
        .wpsc-wrap .wpsc-custom-field-group label { font-weight: 600; display: block; margin-bottom: 4px; }
        .wpsc-wrap .wpsc-custom-field-group input { width: 100%; max-width: 400px; }
        .wpsc-wrap .wpsc-toggle-field, .wpsc-wrap .wpsc-custom-color-field { display: none; }
    ";
    wp_add_inline_style( 'wp-admin', $css );
}
add_action('admin_enqueue_scripts', 'wpsc_admin_enqueue_scripts');

/**
 * Register plugin settings.
 */
function wpsc_register_settings() {
    register_setting( 'wpsc_settings_group', 'wpsc_settings', 'wpsc_sanitize_settings' );
}
add_action( 'admin_init', 'wpsc_register_settings' );

/**
 * Sanitize all settings.
 */
function wpsc_sanitize_settings( $input ) {
    $new_input = [];
    $input = is_array($input) ? $input : [];
    foreach ($input as $key => $val) {
         if (strpos($key, '_url') !== false) $new_input[$key] = esc_url_raw( trim( $val ) );
         else if (strpos($key, '_color') !== false) $new_input[$key] = sanitize_hex_color($val);
         else if ($key === 'icon_size') $new_input[$key] = absint($val);
         else $new_input[$key] = sanitize_text_field( trim( $val ) );
    }
    $new_input['force_display'] = isset($input['force_display']) ? '1' : '0';
    return $new_input;
}

/**
 * Admin page HTML.
 */
function wpsc_admin_page_html() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $settings = wp_parse_args(get_option('wpsc_settings', array()), [
        'position' => 'bottom-left', 'icon_size' => 44, 'icon_shape' => 'circle', 'color_scheme' => 'brand',
        'custom_bg_color' => '#333333', 'custom_icon_color' => '#ffffff', 'display_mode'=>'all', 'toggle_icon' => 'fas fa-phone', 'toggle_direction' => 'upwards'
    ]);
    ?>
    <div class="wrap wpsc-wrap">
        <div class="wpsc-header">
            <img src="https://ehitb.com/wp-content/uploads/2025/10/65149e15-4960-477a-8ac2-a10f2e114e91.png" alt="WP Social Connect">
            <div>
                 <h1><?php esc_html_e( 'WP Social Connect Settings', 'wp-social-connect' ); ?></h1>
                 <p class="wpsc-powered-by"><a href="https://ehitb.com" target="_blank">Powered by Ekramul Hassan Infotech Bangladesh</a></p>
            </div>
        </div>

        <form method="post" action="options.php">
            <?php
            settings_fields( 'wpsc_settings_group' );
            ?>
            
            <h3><?php _e('Social Media Profiles', 'wp-social-connect'); ?></h3>
            <p><?php _e('Enter your profile links below. Only filled fields will be displayed.', 'wp-social-connect'); ?></p>
            <table class="form-table">
               <?php 
               $profiles = ['Facebook'=>'facebook', 'X (Twitter)'=>'twitter', 'Instagram'=>'instagram', 'LinkedIn'=>'linkedin', 'YouTube'=>'youtube', 'WhatsApp'=>'whatsapp', 'Telegram'=>'telegram', 'Email'=>'email'];
               foreach ($profiles as $label => $key) {
                   ?>
                    <tr valign="top">
                        <th scope="row"><label for="wpsc_<?php echo $key; ?>_url"><?php echo $label; ?></label></th>
                        <td><input type="url" id="wpsc_<?php echo $key; ?>_url" name="wpsc_settings[<?php echo $key; ?>_url]" value="<?php echo isset($settings[$key.'_url']) ? esc_attr($settings[$key.'_url']) : ''; ?>" class="regular-text" placeholder="https://<?php echo $key; ?>.com/your-profile" /></td>
                    </tr>
                   <?php
               }
               ?>
            </table>

            <h3><?php _e('Other Contact Links', 'wp-social-connect'); ?></h3>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><label for="wpsc_call_url">Call</label></th>
                    <td><input type="tel" id="wpsc_call_url" name="wpsc_settings[call_url]" value="<?php echo isset($settings['call_url']) ? esc_attr($settings['call_url']) : ''; ?>" class="regular-text" placeholder="tel:+1234567890" /></td>
                </tr>
                <?php for ($i = 1; $i <= 3; $i++): ?>
                    <tr valign="top">
                        <th scope="row"><label for="wpsc_custom<?php echo $i; ?>_url"><?php printf(__('Custom Link %d', 'wp-social-connect'), $i); ?></label></th>
                        <td>
                            <div class="wpsc-custom-field-group">
                                <p>
                                    <label for="wpsc_custom<?php echo $i; ?>_label">Label</label>
                                    <input type="text" id="wpsc_custom<?php echo $i; ?>_label" name="wpsc_settings[custom<?php echo $i; ?>_label]" value="<?php echo isset($settings['custom'.$i.'_label']) ? esc_attr($settings['custom'.$i.'_label']) : ''; ?>" placeholder="e.g., Our Website" />
                                </p>
                                <p>
                                    <label for="wpsc_custom<?php echo $i; ?>_url">URL</label>
                                    <input type="url" id="wpsc_custom<?php echo $i; ?>_url" name="wpsc_settings[custom<?php echo $i; ?>_url]" value="<?php echo isset($settings['custom'.$i.'_url']) ? esc_attr($settings['custom'.$i.'_url']) : ''; ?>" class="regular-text" placeholder="https://example.com" />
                                </p>
                                <p>
                                    <label for="wpsc_custom<?php echo $i; ?>_icon">Font Awesome Icon Class</label>
                                    <input type="text" id="wpsc_custom<?php echo $i; ?>_icon" name="wpsc_settings[custom<?php echo $i; ?>_icon]" value="<?php echo isset($settings['custom'.$i.'_icon']) ? esc_attr($settings['custom'.$i.'_icon']) : ''; ?>" placeholder="e.g., fas fa-globe" />
                                </p>
                                <p class="description">Find icons at <a href="https://fontawesome.com/search?m=free" target="_blank">fontawesome.com</a>. Use the full class name.</p>
                            </div>
                        </td>
                    </tr>
                <?php endfor; ?>
             </table>

            <h3><?php _e('Display & Style Settings', 'wp-social-connect'); ?></h3>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><label for="wpsc_display_mode">Display Mode</label></th>
                    <td>
                        <select id="wpsc_display_mode" name="wpsc_settings[display_mode]">
                            <option value="all" <?php selected($settings['display_mode'], 'all'); ?>>Show All Icons</option>
                            <option value="toggle" <?php selected($settings['display_mode'], 'toggle'); ?>>Toggle on Hover</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top" class="wpsc-toggle-field">
                    <th scope="row"><label for="wpsc_toggle_icon">Toggle Icon</label></th>
                    <td>
                         <select id="wpsc_toggle_icon" name="wpsc_settings[toggle_icon]">
                            <option value="fas fa-phone" <?php selected($settings['toggle_icon'], 'fas fa-phone'); ?>>Call Icon</option>
                            <option value="fas fa-comment-dots" <?php selected($settings['toggle_icon'], 'fas fa-comment-dots'); ?>>Message Icon</option>
                            <option value="fas fa-plus" <?php selected($settings['toggle_icon'], 'fas fa-plus'); ?>>Plus Icon</option>
                            <option value="fas fa-share-alt" <?php selected($settings['toggle_icon'], 'fas fa-share-alt'); ?>>Share Icon</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top" class="wpsc-toggle-field">
                    <th scope="row"><label for="wpsc_toggle_direction">Toggle Direction</label></th>
                    <td>
                         <select id="wpsc_toggle_direction" name="wpsc_settings[toggle_direction]">
                            <option value="upwards" <?php selected($settings['toggle_direction'], 'upwards'); ?>>Expand Upwards</option>
                            <option value="downwards" <?php selected($settings['toggle_direction'], 'downwards'); ?>>Expand Downwards</option>
                         </select>
                         <p class="description">The direction icons expand when hovering.</p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="wpsc_position">Position</label></th>
                    <td>
                        <select id="wpsc_position" name="wpsc_settings[position]">
                            <option value="bottom-left" <?php selected($settings['position'], 'bottom-left'); ?>>Bottom Left</option>
                            <option value="bottom-right" <?php selected($settings['position'], 'bottom-right'); ?>>Bottom Right</option>
                            <option value="top-left" <?php selected($settings['position'], 'top-left'); ?>>Top Left</option>
                            <option value="top-right" <?php selected($settings['position'], 'top-right'); ?>>Top Right</option>
                            <option value="middle-left" <?php selected($settings['position'], 'middle-left'); ?>>Middle Left</option>
                            <option value="middle-right" <?php selected($settings['position'], 'middle-right'); ?>>Middle Right</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="wpsc_icon_size">Icon Size (px)</label></th>
                    <td><input type="number" id="wpsc_icon_size" name="wpsc_settings[icon_size]" value="<?php echo esc_attr($settings['icon_size']); ?>" class="small-text" min="20" max="100" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="wpsc_icon_shape">Icon Shape</label></th>
                    <td>
                         <select id="wpsc_icon_shape" name="wpsc_settings[icon_shape]">
                            <option value="circle" <?php selected($settings['icon_shape'], 'circle'); ?>>Circle</option>
                            <option value="rounded" <?php selected($settings['icon_shape'], 'rounded'); ?>>Rounded Corners</option>
                            <option value="square" <?php selected($settings['icon_shape'], 'square'); ?>>Square</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="wpsc_color_scheme">Color Scheme</label></th>
                    <td>
                        <select id="wpsc_color_scheme" name="wpsc_settings[color_scheme]">
                            <option value="brand" <?php selected($settings['color_scheme'], 'brand'); ?>>Official Brand Colors</option>
                            <option value="custom" <?php selected($settings['color_scheme'], 'custom'); ?>>Custom Colors</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top" class="wpsc-custom-color-field">
                    <th scope="row"><label for="wpsc_custom_bg_color">Custom BG Color</label></th>
                    <td><input type="text" id="wpsc_custom_bg_color" name="wpsc_settings[custom_bg_color]" value="<?php echo esc_attr($settings['custom_bg_color']); ?>" class="wpsc-color-picker" data-default-color="#333333" /></td>
                </tr>
                 <tr valign="top" class="wpsc-custom-color-field">
                    <th scope="row"><label for="wpsc_custom_icon_color">Custom Icon Color</label></th>
                    <td><input type="text" id="wpsc_custom_icon_color" name="wpsc_settings[custom_icon_color]" value="<?php echo esc_attr($settings['custom_icon_color']); ?>" class="wpsc-color-picker" data-default-color="#ffffff" /></td>
                </tr>
            </table>

            <h3><?php _e('Advanced Settings', 'wp-social-connect'); ?></h3>
            <table class="form-table">
                 <tr valign="top">
                    <th scope="row">Force Display Icons</th>
                    <td>
                        <label for="wpsc_force_display">
                            <input type="checkbox" id="wpsc_force_display" name="wpsc_settings[force_display]" value="1" <?php checked($settings['force_display'], '1'); ?> />
                            <span class="description"><?php _e('Enable this if icons are not appearing. It bypasses potential theme conflicts.', 'wp-social-connect'); ?></span>
                        </label>
                    </td>
                </tr>
            </table>
            
            <?php submit_button(); ?>
        </form>
    </div>
    <script>
        jQuery(document).ready(function($) {
            $('.wpsc-color-picker').wpColorPicker();

            function toggleCustomColors() {
                if ($('#wpsc_color_scheme').val() === 'custom') {
                    $('.wpsc-custom-color-field').closest('tr').show();
                } else {
                    $('.wpsc-custom-color-field').closest('tr').hide();
                }
            }
            function toggleToggleFields() {
                if ($('#wpsc_display_mode').val() === 'toggle') {
                     $('.wpsc-toggle-field').closest('tr').show();
                } else {
                     $('.wpsc-toggle-field').closest('tr').hide();
                }
            }

            toggleCustomColors();
            toggleToggleFields();
            $('#wpsc_color_scheme').on('change', toggleCustomColors);
            $('#wpsc_display_mode').on('change', toggleToggleFields);
        });
    </script>
    <?php
}


/**
 * Enqueue frontend scripts and styles.
 */
function wpsc_enqueue_frontend_assets() {
    wp_enqueue_style( 'font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css', array(), '5.15.4' );
    wp_add_inline_style( 'font-awesome', wpsc_get_frontend_css() );
}

/**
 * Get Frontend CSS.
 */
function wpsc_get_frontend_css() {
    $opts = wp_parse_args(get_option('wpsc_settings'), [
        'position' => 'bottom-left', 'icon_size' => 44, 'icon_shape' => 'circle', 'color_scheme' => 'brand',
        'custom_bg_color' => '#333333', 'custom_icon_color' => '#ffffff', 'display_mode'=>'all', 'toggle_direction' => 'upwards'
    ]);
    
    $pos_css = '';
    switch ($opts['position']) {
        case 'top-left': $pos_css = 'top: 15px; left: 15px;'; break;
        case 'top-right': $pos_css = 'top: 15px; right: 15px;'; break;
        case 'bottom-right': $pos_css = 'bottom: 15px; right: 15px;'; break;
        case 'middle-left': $pos_css = 'top: 50%; left: 15px; transform: translateY(-50%);'; break;
        case 'middle-right': $pos_css = 'top: 50%; right: 15px; transform: translateY(-50%);'; break;
        default: $pos_css = 'bottom: 15px; left: 15px;'; break; // bottom-left
    }
    
    $size = absint($opts['icon_size']);
    $size_css = "width: {$size}px; height: {$size}px; font-size: ".round($size * 0.5)."px;";
    $shape_css = 'border-radius: '.($opts['icon_shape']==='circle'?'50%':($opts['icon_shape']==='rounded'?'8px':'0')).';';
    
    $brand_colors = ".wpsc-icon-facebook{background-color:#1877F2}.wpsc-icon-twitter{background-color:#14171A}.wpsc-icon-instagram{background:radial-gradient(circle at 30% 107%,#fdf497 0%,#fd5949 45%,#d6249f 60%,#285AEB 90%)}.wpsc-icon-linkedin{background-color:#0A66C2}.wpsc-icon-youtube{background-color:#FF0000}.wpsc-icon-whatsapp{background-color:#25D366}.wpsc-icon-telegram{background-color:#2AABEE}.wpsc-icon-email{background-color:#888888}.wpsc-icon-call{background-color:#4CAF50}.wpsc-icon-custom1,.wpsc-icon-custom2,.wpsc-icon-custom3{background-color:#607D8B}";
    $custom_color_css = ($opts['color_scheme']==='custom') ? "background-color:".sanitize_hex_color($opts['custom_bg_color'])."; color:".sanitize_hex_color($opts['custom_icon_color']).";" : 'color: #fff;';

    $base_css = ".wpsc-social-container{position:fixed;{$pos_css}z-index:99999!important}.wpsc-social-container a, .wpsc-toggle-button{display:flex;align-items:center;justify-content:center;text-decoration:none;transition:all .2s ease-in-out;box-shadow:0 2px 6px rgba(0,0,0,.25);{$size_css}{$shape_css}{$custom_color_css}}.wpsc-social-container a:hover, .wpsc-toggle-button:hover{transform:scale(1.1);box-shadow:0 4px 10px rgba(0,0,0,.3)}";

    if ($opts['display_mode'] === 'toggle') {
        $flex_direction_css = ($opts['toggle_direction'] === 'downwards') ? 'column' : 'column-reverse';
        $align_items_css = (strpos($opts['position'],'left')!==false?'flex-start':'flex-end');
        $margin_css = ($opts['toggle_direction'] === 'downwards') ? 'margin-top:8px' : 'margin-bottom:8px';

        $toggle_css = ".wpsc-social-container.wpsc-mode-toggle{display:flex;flex-direction:{$flex_direction_css};align-items:{$align_items_css}} .wpsc-icon-list{display:flex;flex-direction:column;gap:8px;{$margin_css}} .wpsc-icon-list a{opacity:0;visibility:hidden;transform:scale(0);transition:opacity .2s, visibility .2s, transform .2s ease-out} .wpsc-social-container.wpsc-mode-toggle:hover .wpsc-icon-list a{opacity:1;visibility:visible;transform:scale(1)} .wpsc-toggle-button{cursor:pointer; background-color: #1877F2 !important; color: #ffffff !important;}";
        for ($i=1; $i<=12; $i++) { $toggle_css .= ".wpsc-social-container.wpsc-mode-toggle:hover .wpsc-icon-list a:nth-child({$i}){transition-delay:".($i*0.05)."s}"; }
        return $base_css . $toggle_css . ($opts['color_scheme']==='brand'?$brand_colors:'');
    } else {
        $all_mode_css = ".wpsc-social-container.wpsc-mode-all {display: flex; flex-direction: column; gap: 8px;}";
        return $base_css . $all_mode_css . ($opts['color_scheme']==='brand'?$brand_colors:'');
    }
}


/**
 * Get Rendered HTML for icons.
 */
function wpsc_get_social_icons_html() {
    $settings = get_option('wpsc_settings', array());
    if (empty($settings)) return '';
    
    $all_links = [];
    $social_icon_map = ['facebook'=>'fab fa-facebook-f','twitter'=>'fab fa-twitter','instagram'=>'fab fa-instagram','linkedin'=>'fab fa-linkedin-in','youtube'=>'fab fa-youtube','whatsapp'=>'fab fa-whatsapp','telegram'=>'fab fa-telegram-plane','email'=>'fas fa-envelope'];

    foreach($social_icon_map as $id => $icon) {
        if (!empty($settings[$id . '_url'])) {
            $all_links[] = ['url' => $settings[$id.'_url'], 'id' => $id, 'label' => ucfirst($id), 'icon' => $icon];
        }
    }
    if (!empty($settings['call_url'])) {
        $all_links[] = ['url' => $settings['call_url'], 'id' => 'call', 'label' => 'Call Us', 'icon' => 'fas fa-phone'];
    }
    for ($i=1; $i<=3; $i++) {
        if (!empty($settings["custom{$i}_url"]) && !empty($settings["custom{$i}_icon"])) {
            $all_links[] = ['url' => $settings["custom{$i}_url"], 'id' => "custom{$i}", 'label' => ($settings["custom{$i}_label"] ?? "Custom Link {$i}"), 'icon' => $settings["custom{$i}_icon"]];
        }
    }

    if (empty($all_links)) return '';

    $links_html = '';
    foreach ($all_links as $link) {
        $links_html .= sprintf('<a href="%s" target="_blank" rel="noopener noreferrer" class="wpsc-icon-%s" aria-label="%s"><i class="%s"></i></a>', esc_url($link['url']), esc_attr($link['id']), esc_attr($link['label']), esc_attr($link['icon']));
    }

    $display_mode = $settings['display_mode'] ?? 'all';
    $container_classes = 'wpsc-social-container';
    $inner_html = '';

    if ($display_mode === 'toggle') {
        $container_classes .= ' wpsc-mode-toggle';
        $toggle_icon = $settings['toggle_icon'] ?? 'fas fa-phone';
        $toggle_button_html = sprintf('<div class="wpsc-toggle-button"><i class="%s"></i></div>', esc_attr($toggle_icon));
        $inner_html = sprintf('<div class="wpsc-icon-list">%s</div>%s', $links_html, $toggle_button_html);
    } else {
        $container_classes .= ' wpsc-mode-all';
        $inner_html = $links_html;
    }
    return sprintf('<div class="%s">%s</div>', $container_classes, $inner_html);
}


/**
 * Render the social icons on the frontend.
 */
function wpsc_render_social_icons() {
    echo wpsc_get_social_icons_html();
}

/**
 * Hook to display icons.
 */
function wpsc_output_icons() {
    $settings = get_option('wpsc_settings');
    $force_display = isset($settings['force_display']) && $settings['force_display'] == '1';

    if (!$force_display) {
        add_action('wp_enqueue_scripts', 'wpsc_enqueue_frontend_assets');
        add_action('wp_footer', 'wpsc_render_social_icons');
    } else {
        add_action('template_redirect', 'wpsc_buffer_start', -1);
    }
}
add_action('wp', 'wpsc_output_icons');

/**
 * Force render icons using output buffering.
 */
function wpsc_force_render_icons_buffer($buffer) {
    if (strpos($buffer, '</body>') === false) return $buffer;
    $styles = wpsc_get_frontend_css();
    $icons_html = wpsc_get_social_icons_html();
    $injection = "<link rel='stylesheet' id='font-awesome-css' href='https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css' type='text/css' media='all' /><style type='text/css'>{$styles}</style>";
    $buffer = str_replace('</head>', $injection . '</head>', $buffer);
    $buffer = str_replace('</body>', $icons_html . '</body>', $buffer);
    return $buffer;
}

// Start output buffering for the force render method
function wpsc_buffer_start() {
    ob_start("wpsc_force_render_icons_buffer");
}

